﻿cordova.define("diloc-fileviewer.FullTextSearch", function(require, exports, module) {
/**
 * @file
 * @version 0.1
 * @copyright 2017 CN-Consult GmbH
 * @author Marcel Bohländer <marcel.bohlaender@cn-consult.eu>
 */
var exec = require("cordova/exec");
/**
 * This FullTextSearch class uses the possibility of the Foxit MobileSDK to create an indexed search database for a full text search.
 * Use indexDirectory function for indexing the whole sync root of the user and the indexFile function for adding newly added files to the index database.
 * But be carefully, if you call indexDirectory or indexFile twice for the same directory/ files you'll get duplicate results.
 * To avoid this you better should call indexDirectory periodically with reUpdate set to true, for setting up a fresh index database.
 * While the index process is running you can call "getIndexProgress" for getting the current index progress an integer between 0 and 100.
 * If you want to search for a given search word you'll get an array of result objects back.
 * For e.g.:
 * results: [{
 *      filePath: <path-to-found-file>,
 *      page: <page-on-which-search-word-was-found>,
 *      searchPrefix: <x-chars-before-search-word>,
 *      searchSuffix: <x-chars-after-search-word>
 * },{
 *      filePath: <path-to-found-file>,
 *      page: <page-on-which-search-word-was-found>,
 *      searchPrefix: <x-chars-before-search-word>,
 *      searchSuffix: <x-chars-after-search-word>
 * }]
 * You are also able to abort by ("abortIndexing") a running indexing process or delete the current search database by calling "deleteDatabase".
 *
 * @param {Function} _callback The callback function that should be called if FullTextSearch has been constructed.
 * @param {Object} _scope The scope in which the callback should be called. (optional)
 */
FullTextSearch = function(_callback, _scope)
{
	/**
	 * FullTextSearch constructor.
	 *
	 * @param {Function} _callback The callback function that should be called if FullTextSearch has been constructed.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.construct = function(_callback, _scope)
	{
		exec(function() {
			console.log("FullTextSearch initialized!");
			Ext.callback(_callback,_scope);
		},this.onCordovaExecFail,"FullTextSearch","initializeFullTextSearch",[]);
	};

	/**
	 * Indexes all pdf files of a whole given directory.
	 * Do NOT call it twice, otherwise you'll get all results twice too.
	 * If you want to re initialize search result database, set reUpdate to true.
	 *
	 * @param {String} _directoryPath The path to the directory of which the pdf files should be indexed.
	 * @param {Boolean} _reUpdate Set to true, if you want to re update your index database.
	 * @param {Function} _callback The callback function that should be called if directory indexing has been finished.
	 *                              It will be called with a single boolean param for giving the information of indexing was successfully or not.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.indexDirectory = function(_directoryPath, _reUpdate, _callback, _scope)
	{
		console.log("Indexing directory: ",_directoryPath, _reUpdate);
		exec(function(_success) {
			Ext.callback(_callback,_scope,[_success]);
		},this.onCordovaExecFail,"FullTextSearch","indexDirectory",[_directoryPath, _reUpdate]);
	};

	/**
	 * Indexes a given (pdf) file.
	 * Do NOT call it twice for the same file, otherwise you'll get duplicate results for this file.
	 *
	 * @param {String} _filePath The path to the (pdf) file which should be indexed.
	 * @param {Function} _callback The callback function that should be called if file indexing has been finished.
	 *                              It will be called with a single boolean param for giving the information of indexing was successfully or not.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.indexFile = function(_filePath, _callback, _scope)
	{
		console.log("Indexing file: ",_filePath);
		exec(function() {
			console.log("Indexing of file "+_filePath+" was successfully!");
			Ext.callback(_callback,_scope,[true]);
		},function() {
			console.log("Could not index file "+_filePath);
			Ext.callback(_callback,_scope,[false]);
		},"FullTextSearch","indexFile",[_filePath]);
	};

	/**
	 * Searches through the index database for the given searchWord.
	 *
	 * @param {String} _searchWord The word for which should been searched.
	 * @param {int} _prefixSuffixLength The amount of chars of before and after the search word which should be returned too.
	 * @param {Function} _callback The callback function which should be called if the search has been finished.
	 *                              It will be called with a results array. This array contains for each search result an object with all necessary data (see class doc).
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.search = function(_searchWord, _prefixSuffixLength,  _callback, _scope)
	{
		console.log("Searching for ",_searchWord);
		exec(function(_results) {
			Ext.callback(_callback,_scope,[_results]);
		},this.onCordovaExecFail,"FullTextSearch","search",[_searchWord, _prefixSuffixLength]);
	};

	/**
	 * Deletes the search database.
	 *
	 * @param {Function} _callback The callback function which should be called if the deletion of the database has been finished.
	 *                              It will be called with a single boolean param for giving the information of deletion was successfully or not.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.deleteDatabase = function(_callback, _scope)
	{
		console.log("Deleting search database...");
		exec(function(_success) {
			Ext.callback(_callback,_scope,[_success]);
		},this.onCordovaExecFail,"FullTextSearch","deleteDatabase");
	};

	/**
	 * Aborts the running index process of a directory and deletes the search database.
	 *
	 * @param {Function} _callback The callback function which should be called if the index process has been aborted.
	 *                              It will be called with a single boolean param for giving the information whether indexing has been successfully aborted or not.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.abortIndexing = function(_callback, _scope)
	{
		console.log("Aborting indexing...");
		exec(function(_success) {
			Ext.callback(_callback,_scope,[_success]);
		},this.onCordovaExecFail,"FullTextSearch","abortIndexing");
	};

	/**
	 * Returns the current index progress. It will return 100, if indexing has been finished and 0 if no index process is currently running.
	 *
	 * @param {Function} _callback The callback function which should be called if an index progress is available.
	 *                              It will be called with a single integer param for returning the progress.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.getIndexProgress = function(_callback, _scope)
	{
		console.log("Retrieving indexing progress...");
		exec(function(_progress) {
			Ext.callback(_callback,_scope,[_progress]);
		},this.onCordovaExecFail,"FullTextSearch","getIndexProgress");
	};

	/**
	 * This is a generic callback function if a cordova exec call fails.
	 *
	 * @param {string} _error The message of the failed cordova exec call.
	 */
	this.onCordovaExecFail=function(_error)
	{
		console.log("Cordova exec failed: "+ _error);
	};

	this.construct(_callback, _scope);
};

module.exports = FullTextSearch;

});
