﻿cordova.define("diloc-fileviewer.diloc-fileviewer", function(require, exports, module) {
/**
 * @file
 * @version 0.1
 * @copyright 2017 CN-Consult GmbH
 * @author Wladimir Tarasov <wladimir.tarasov@cn-consult.eu>
 */
var exec = require("cordova/exec");
/**
 *
 * This plugin offers the possibility to open files (chiefly pdf files, other file types listed below) on iOS and android.
 * A file can be opened in fullscreen mode or in a given rect, which is defined by the upper left corner, the height and the width.
 * This fileviewer can be instanced multiple times, which is useful for showing two or more files side-by-side.
 *
 * Supported file types
 * Android:
 * - pdf
 * - html
 * - images (jpeg, png, bmp, ...)
 *
 * iOS:
 * - pdf
 * - html
 * - images (jpeg, png, bmp, ...)
 * - doc, docx
 * - xls, xlsx
 * - ppt, pptx
 * - txt
 *
 * @param {Function} _callback The callback function that should be called if FileViewerNG has been constructed.
 * @param {Object} _scope The scope in which the callback should be called. (optional)
 */
FileViewerNG = function (_callback, _scope)
{
	this.id = null;
	this.serviceName = "FileViewer";

	/**
	 * FileViewerNG constructor.
	 *
	 * @param {Function} _callback The callback function that should be called if FileViewerNG has been constructed.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.construct = function (_callback, _scope)
	{
		if (typeof device == 'undefined')
		{ //we are not on phonegap, so make as us at least not complain with a dirty hack...
			window.cordova = {
				exec: function ()
				{
				}
			};
		}

		var me = this;
		exec(function (_id)
		{
			console.log("Set id:" + _id + " for this fileviewer.");
			me.id = _id;
			Ext.callback(_callback, _scope);
		}, this.onCordovaExecFail, this.serviceName, "createViewer", []);
	};

	/**
	 * This is a generic callback function if a cordova exec call fails.
	 *
	 * @param {string} _error The message of the failed cordova exec call.
	 */
	this.onCordovaExecFail = function (_error)
	{
		console.log("Cordova exec fail: " + _error);
	};

	/**
	 * This is a generic callback function for successfully cordova exec calls.
	 */
	this.onCordovaExecSuccess = function ()
	{

	};

	/**
	 * Sets the rect (x, y, width and height) in which this FileViewer should open files.
	 *
	 * @param {Object} _rect An object with the members x, y, width and height of the rect in which this FileViewer should open files.
	 */
	this.setRect = function (_rect)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.setRect() --> Sets a given rect to the FileViewer with the id " + this.id);
		exec(function ()
		{
			Ext.fireEvent("fileviewerrectconfigured");
		}, this.onCordovaExecFail, this.serviceName, "setRect", [this.id, _rect.x, _rect.y, _rect.width, _rect.height]);
	};

	/**
	 * Returns the current rect of the file-viewer to the supplied callback function.
	 *
	 * @param {Function} _callback The callback function that should be called to deliver the result.
	 *                             The callback is called with a single parameter that is an object which contains the information
	 *                             about the current rect in the same format as setRect() expects it.
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.getRect = function (_callback, _scope)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}

		console.log("FileViewer.getRect() --> Gets a given rect with the id " + this.id);
		var me = this;
		exec(function (_rect)
		{
			console.log("GetRect with id:" + me.id + " - ", _rect);
			Ext.callback(_callback, _scope, [_rect]);
		}, this.onCordovaExecFail, this.serviceName, "getRect", [this.id]);
	};

	/**
	 * Shows the FileViewer into the given rect.
	 */
	this.show = function ()
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.show() --> Shows the FileViewer with the id " + this.id);
		exec(function ()
		{
			Ext.fireEvent("fileviewershown");
		}, this.onCordovaExecFail, this.serviceName, "show", [this.id]);
	};

	/**
	 * Hides the FileViewer.
	 */
	this.hide = function ()
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.hide() --> Hides the FileViewer with the id " + this.id);
		exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "hide", [this.id]);
	};

	/**
	 * Delivers true/false to the supplied callback depending on the visibility of the FileViewer.
	 * See show() and hide().
	 *
	 * @param {Function} _callback The callback function that should be called to deliver the result.
	 *                             The callback is called with a single parameter that is a boolean true if the viewer is visible,
	 *                             false otherwise.
	 *
	 * @param {Object} _scope The scope in which the callback should be called. (optional)
	 */
	this.isVisible = function (_callback, _scope)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}

		console.log("FileViewer.isVisible() --> Checks visibility of the FileViewer with the id " + this.id);
		cordova.exec(function (_visible)
		{
			console.log("Viewer is visible: ", _visible);
			if (_visible) Ext.callback(_callback, _scope, [true]);
			else Ext.callback(_callback, _scope, [false]);
		}, this.onCordovaExecFail, this.serviceName, "isVisible", [this.id]);
	};

	/**
	 * Shows the FileViewer in the fullscreen mode.
	 */
	this.enterFullscreen = function ()
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.enterFullscreen() --> Enter fullscreen mode of the FileViewer with the id " + this.id);
		exec(function ()
		{
			Ext.fireEvent("fullscreenleft");
		}, this.onCordovaExecFail, this.serviceName, "enterFullscreen", [this.id]);
	};

	/**
	 * Exits the fullscreen mode of this FileViewer.
	 */
	this.exitFullscreen = function ()
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		this.isVisible(function (_visible)
		{
			if (_visible)
			{
				console.log("FileViewer.exitFullscreen() --> Exit fullscreen mode of the FileViewer with the id " + this.id);
				exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "exitFullscreen", [this.id]);
			}
			else
			{
				console.log("FileViewer.exitFullscreen() --> not visible do nothing");
			}
		}, this);

	};

	/**
	 * Loads a file into this FileViewer and shows it in the configured rect.
	 *
	 * @param {String} _filePath The path to the file that should be displayed. This can either be a local file:/// url or a remote http:// url.
	 * @param {String} _onPage Opens a document on the given page. This works only for pdf documents.
	 */
	this.loadFile = function (_filePath, _onPage)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		if (typeof _onPage == 'undefined') _onPage = 0;

		console.log("FileViewer.loadFile() --> Load file " + _filePath + " into the FileViewer with the id " + this.id);
		exec(function ()
		{
			Ext.fireEvent("fileviewerfileloaded");
		}, this.onCordovaExecFail, this.serviceName, "loadFile", [this.id, _filePath, _onPage]);
	};

	/**
	 * Returns the page of the current pdf file which is loaded in our file viewer (first pdf page  = value 0,images and other stuff = value 0).
	 */
	this.getCurrentPage = function (_callback, _scope)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}

		console.log("FileViewer.getCurrentPage() --> Get current page with the id " + this.id);
		var me = this;
		exec(function (_currentPage)
		{
			console.log("FileViewer with the id " + me.id + " has opened a file on page " + _currentPage);
			Ext.callback(_callback, _scope, [_currentPage]);
		}, this.onCordovaExecFail, this.serviceName, "getCurrentPage", [this.id]);
	};

	/**
	 * Enable nightmode for this FileViewer.
	 *
	 * @param {Boolean} _enableNightMode True for enabling nightmode mode, false for disabling.
	 */
	this.enableNightMode = function (_enableNightMode)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.enableNightMode() --> enable nightmode for FileViewer with the id " + this.id);
		exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "enableNightMode", [this.id, _enableNightMode]);
	};

	/**
	 * Toggle horizontal/vertical scrolling for this FileViewer.
	 */
	this.toggleScrolling = function ()
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.toggleScrolling() --> toggle scrolling for FileViewer with the id " + this.id);
		exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "toggleScrolling", [this.id]);
	};

	/**
	 * Enables the default foxit toolbar for this FileViewer.
	 *
	 * @param {Boolean} _enableToolbar True for enabling toolbar, false for disabling.
	 */
	this.enableToolbar = function (_enableToolbar)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.enableToolbar() --> enable toolbar for FileViewer with the id " + this.id);
		exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "enableToolbar", [this.id, _enableToolbar]);
	};

	/**
	 * Cleans up all fdf files, for which no corresponding file exists anymore.
	 *
	 * @param {String} _basePath Path to the directory in which it will search for lonely fdf files.
	 */
	this.cleanupPDFAnnotations = function (_basePath)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.cleanupPDFAnnotations() --> clean all fdf files.");
		exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "cleanupPDFAnnotations", [this.id, _basePath]);
	};

	/**
	 * Deletes a fdf file.
	 *
	 * @param {String} _path Path to the file.
	 */
	this.deleteFDFFile = function (_path)
	{
		if (!this.id)
		{
			console.log("We have no FileViewer ID, cannot do anything!");
			return;
		}
		console.log("FileViewer.deleteFDFFile(" + _path + ") --> clean fdf file.");
		exec(this.onCordovaExecSuccess, this.onCordovaExecFail, this.serviceName, "deleteFDFFile", [this.id, _path]);
	};

	this.construct(_callback, _scope);
};

module.exports = FileViewerNG;

});
